unit XFileUtl;

{
  ===========================
  eXtended File Utilities 1.2 (1999-09-24)
  ===========================
  Routines for working width files, directories and file names.

  Freeware.

  Copyright  Roman Stedronsky 1998, Roman.Stedronsky@seznam.cz

  All rights reserved. You may use this software in an application
  without fee or royalty, provided this copyright notice remains intact.
}

interface

uses Windows;

{ file management routines }
function DirectoryExists(const Name: string): boolean; // test existence of directory
function ForceDirectories(Dir: string): boolean; // create all directories in path
procedure ClearDir(Dir: string); // deletes the files in an existing directory
procedure FileCopy(const SourceFilename, TargetFilename: string); overload; // shell file copy operation
function FileCopy(const SourceFilename, TargetFilename: string; const FailIfExists: boolean): boolean; overload; // shell file copy operation
function ValidFileName(const FileName: string): boolean;

{ shell management routines }
procedure RunDosCommand(CmdLine: string); // runs dos command - explicitly calling command.com
function ExecuteFile(const FileName, Params, DefaultDir: string;
  ShowCmd: Integer): THandle; // open file

{ filename handling routines}
function Slash(const DirName: string): string; overload; // if not '\' at the end add it
function Slash(const DirName, FileName: string): string; overload; // compose DirName\Filename
function RemoveSlash(const DirName: string): string; // remove '\' from the end
function ExtractFileNameOnly(const FileName: string): string; // filename without dot and extension
function ExtractFileExtOnly(const FileName: string): string; // extension without dot
function ExcludeFileExt(const FileName: string): string; // full path and name without dot and extension
function ExpandRelativePath(const BasePath, RelativeName: string): string; // full path name from base path & relative path
function ComposePath(const PathFileName, FileName: string): string; // extract path from first and add filename from second
function AddProgramPath(const FileName: string): string; // adds program path to filename
procedure FileNameSplit(Spec: string; var Path, Name, Ext: string); // split fullname into parts: path, name and extension
function ProcessPath(const EditText: string; var Drive: Char;
  var DirPart: string; var FilePart: string): boolean; // split fullname into: drive, path, name with extension

{ other string routines}
procedure TrimVar(var s: string); // remove spaces from string

implementation

uses
  SysUtils, Classes, ShellAPI;

{ file management routines }

{$R-}

function DirectoryExists(const Name: string): Boolean;
var
  Code: Integer;
begin
  Code := GetFileAttributes(PChar(Name));
  Result := (Code <> -1) and (FILE_ATTRIBUTE_DIRECTORY and Code <> 0);
end;
{$R+}

function ForceDirectories(Dir: string): boolean;
begin
  Result := false;
  if not (Length(Dir) = 0) then
//    raise Exception.Create('Cannot create directory');//SCannotCreateDir);
  begin
    if (AnsiLastChar(Dir) <> nil) and (AnsiLastChar(Dir)^ = '\') then
      Delete(Dir, Length(Dir), 1);
    if (Length(Dir) < 3) or DirectoryExists(Dir) or (ExtractFilePath(Dir) = Dir) then Exit; // avoid 'xyz:\' problem.
    ForceDirectories(ExtractFilePath(Dir));
    Result := CreateDir(Dir);
  end;
end;

procedure ClearDir(Dir: string); { Deletes the files in an existing directory }
var
  CurrentDir: string;
  SearchRec: TSearchRec;
  Result: longint;
begin { ClearDir }
  Getdir(0, CurrentDir);
  Chdir(Dir);
  Result := FindFirst(Dir + '\*.*', faAnyFile, SearchRec);
  while Result = 0 do
  begin
    SysUtils.DeleteFile(SearchRec.name);
    Result := FindNext(SearchRec);
  end;
  SysUtils.FindClose(SearchRec);
  Chdir(CurrentDir);
end; { ClearDir }

procedure FileCopy(const SourceFilename, TargetFilename: string);
var
  S, T: TFileStream;
begin
  S := TFileStream.Create(SourceFilename, fmOpenRead);
  try
    T := TFileStream.Create(TargetFilename, fmOpenWrite or fmCreate);
    try
      T.CopyFrom(S, S.Size);
    finally
      T.Free;
    end;
  finally
    S.Free;
  end;
end;

function FileCopy(const SourceFilename, TargetFilename: string; const FailIfExists: boolean): boolean; // shell file copy operation
begin
  Result := CopyFile(PChar(SourceFilename), PChar(TargetFilename), FailIfExists);
end;

function ValidFileName(const FileName: string): Boolean;
  function HasAny(const Str, Substr: string): Boolean;
  var
    I: Integer;
  begin
    Result := False;
    for I := 1 to Length(Substr) do begin
      if Pos(Substr[I], Str) > 0 then begin
        Result := True;
        Break;
      end;
    end;
  end;
begin
  Result := (FileName <> '') and (not HasAny(FileName, '\/:*?"<>|'));
  if Result then Result := Pos('\', ExtractFileName(FileName)) = 0;
end;

{ shell management routines }

procedure RunDosCommand(CmdLine: string);
var
  Result: Integer;
  function LeftStr(s: string; count: Integer): string;
  begin
    LeftStr := Copy(s, 1, count)
  end;
begin
  CmdLine := 'command.com' + ' /C ' + Cmdline + Chr(0);
  Result := WinExec(@CmdLine[1], sw_minimize);
  if Result < 32 then
    MessageBox(0, PChar('Execution of command line' + #13 + LeftStr(CmdLine, Length(CmdLine) - 1) + #13 +
      'failed. Error code:' + IntToStr(Result)), 'Error', mb_IconStop);
end; { RunDosCommand }

function ExecuteFile(const FileName, Params, DefaultDir: string; ShowCmd: Integer): THandle;
var
  zFileName, zParams, zDir: array[0..MAX_PATH] of Char;
begin
  Result := ShellExecute(MainInstance, nil,
    StrPCopy(zFileName, FileName), StrPCopy(zParams, Params),
    StrPCopy(zDir, DefaultDir), ShowCmd);
end;

{ filename handling routines}

function Slash(const DirName: string): string;
begin
  if AnsiLastChar(DirName)^ <> '\' then
    Result := DirName + '\'
  else
    Result := DirName;
end;

function Slash(const DirName, FileName: string): string;
begin
  if (DirName <> '') and (AnsiLastChar(DirName)^ <> '\') then
    Result := DirName + '\' + FileName
  else
    Result := DirName + FileName;
end;

function RemoveSlash(const DirName: string): string;
begin
  if (Length(Dirname) = 0) or ((Length(Dirname) = 3) and (DirName[2] = ':')) or (DirName[Length(Dirname)] <> '\') then
    Result := Dirname
  else
    Result := Copy(Dirname, 1, Length(Dirname) - 1)
end;

function ExtractFileNameOnly(const FileName: string): string;
var
  i, ii: Integer;
begin
  i := Length(FileName);
  while (i > 0) and not (FileName[i] in ['.', '\', ':']) do
    Dec(i);
  if i > 0 then
  begin
    if FileName[i] = '.' then
    begin
      ii := i - 1;
      while (i > 0) and not (FileName[i] in ['\', ':']) do
        Dec(i);
    end
    else
      ii := Length(FileName);
    Result := Copy(FileName, i + 1, ii - i);
  end
  else
    Result := FileName;
end;

function ExtractFileExtOnly(const FileName: string): string;
begin
  Result := ExtractFileExt(FileName);
  if Result <> '' then Result := Copy(Result, 2, Length(Result));
end;

function ExcludeFileExt(const FileName: string): string;
var
  Index: Integer;
begin
  Index := Length(FileName);
  while (Index > 0) and not (FileName[Index] = '.') do
    Dec(Index);
  if Index > 0 then
    Result := Copy(FileName, 1, Index - 1)
  else
    Result := FileName;
end;

function ExpandRelativePath(const BasePath, RelativeName: string): string;
var
  FName: PChar;
  Buffer: array[0..MAX_PATH - 1] of Char;
begin
  SetString(Result, Buffer, GetFullPathName(PChar(Slash(BasePath) + RelativeName), SizeOf(Buffer),
    Buffer, FName));
end;

function ComposePath(const PathFileName, FileName: string): string;
begin
  Result := ExtractFilePath(PathFileName) + ExtractFileName(FileName);
end;

function AddProgramPath(const FileName: string): string;
begin
  Result := Slash(ExtractFilePath(ParamStr(0))) + FileName;
end;

procedure FileNameSplit(Spec: string; var Path, Name, Ext: string);
var
  Len: longint;
  FullName: string;
begin
  Path := ExtractFilePath(Spec);
  Ext := ExtractFileExt(Spec);
  FullName := ExtractFileName(Spec);
  Len := Length(FullName) - Length(Ext);
  Name := Copy(FullName, 1, Len);
end;

function ProcessPath(const EditText: string; var Drive: Char; var DirPart: string; var FilePart: string): boolean;
var
  SaveDir, Root: string;
begin
  SaveDir := GetCurrentDir;
  Drive := SaveDir[1];
  DirPart := EditText;
  if (Length(DirPart) >= 2) and (DirPart[1] = '[') and
    (AnsiLastChar(DirPart)^ = ']') then
    DirPart := Copy(DirPart, 2, Length(DirPart) - 2)
  else
  begin
    Root := ExtractFileDrive(DirPart);
    if Root = '' then
      Root := ExtractFileDrive(SaveDir)
    else
      Delete(DirPart, 1, Length(Root));
    if (Length(Root) >= 2) and (Root[2] = ':') then
      Drive := Root[1]
    else
      Drive := #0;
  end;
  Result := false;
  try
    SetCurrentDir(Root);
    FilePart := ExtractFileName(DirPart);
    if Length(DirPart) = (Length(FilePart) + 1) then
      DirPart := '\'
    else if Length(DirPart) > Length(FilePart) then
      SetLength(DirPart, Length(DirPart) - Length(FilePart) - 1)
    else
    begin
      DirPart := GetCurrentDir;
      Delete(DirPart, 1, Length(ExtractFileDrive(DirPart)));
      if DirPart = '' then
	DirPart := '\';
    end;
    if DirPart <> '' then
      if not SetCurrentDir(DirPart) then Exit;
    if (FilePart <> '') and not
      (((Pos('*', FilePart) > 0) or (Pos('?', FilePart) > 0)) or
      FileExists(FilePart)) then
    begin
      if not SetCurrentDir(FilePart) then Exit;
      if Length(DirPart) = 1 then
	DirPart := '\' + FilePart
      else
	DirPart := DirPart + '\' + FilePart;
      FilePart := '';
    end;
    if Drive = #0 then
      DirPart := Root + DirPart;
    Result := true;
  finally
    SetCurrentDir(SaveDir); { restore original directory }
  end;
end;

{ other string routines}

procedure TrimVar(var s: string);
begin
  while Pos(' ', s) = 1 do
    system.Delete(s, 1, 1);
  while (s <> '') and (s[Length(s)] = ' ') do
    system.Delete(s, Length(s), 1);
end;

end.

